      *  This demonstrates a simple SSL server program.  It creates a
      *  TCP server that waits on port 8765. When it gets a new
      *  connection, it creates a new job (using the spawn() API) and
      *  that job negotiates an SSL session and uses that SSL session
      *  to provide customer info to the SSLCLIENT program.
      *                                   Scott Klement, Sept 2006
      *
      *  To Compile:
      *    - Make sure you have the SOCKET_H, GSKSSL_H,SPAWN_H,SIGNAL_H
      *      and ERRNO_H source members in a QRPGLESRC file in your
      *      library list.
      *    - Make sure you have the CUSTFILE source member uploaded
      *      and in a QDDSSRC file in your library list, and key some
      *      test info into this member.
      *    - Update the PROGRAM and LIBRARY constants (below) to reflect
      *      the location where you put this program on your system.
      *    - type:
      *      CRTPF FILE(CUSTFILE) SRCFILE(xxx/QDDSSRC)
      *    - type:
      *      CRTBNDRPG PGM(SSLSERVER) SRCFILE(xxx/QRPGLESRC) DBGVIEW(*LIST)
      *
      *   Before running:
      *    - Configure an application profile for application id
      *      KLEMENT_SSLDEMO_SSLSERVER in the Digital Certificate Manager
      *    - To do that:
      *        a) Connect to the DCM with your web browser on port 2001
      *               http://my-i5.example.com:2001
      *        b) Choose "Digital Certificate Manager"
      *        c) Select a certificate store, preferably *SYSTEM,
      *               and type the password.
      *        d) On the left, select "Manage Applications"
      *        e) Then select "Add Application"
      *        f) This is a server application. Make sure you use
      *               KLEMENT_SSLDEMO_SSLSERVER as the application id
      *               (or change this code to use the appid that you
      *               provided!)
      *        g) After you've added the application profile, be sure
      *           to use the "update certificate assignment" option
      *           of the digital certificate manager to assign a
      *           certificate to the server.
      *
      *    If you have trouble getting this running, a great place to go
      *    for help is the iSeries Network Forums:
      *          http://www.iseriesnetwork.com/isnetforums/
      *
      *    To run:
      *       SBMJOB CMD(CALL SSLSERVER) JOB(SSLSERVER) JOBQ(QSYSNOMAX)
      *
      *    To end the server:
      *       WRKACTJOB SBS(QSYSWRK)
      *       find the SSLSERVER job and put a "4" next to it.  You can
      *       end it *CNTRLD, it will detect that you asked it to end
      *       within 15 seconds, and shut down gracefully.
      *
      *    To see if it works:
      *       Run the SSLCLIENT program. See the comments in that
      *       source member for more info.
      *
     H DFTACTGRP(*NO) BNDDIR('QC2LE')
     FCUSTFILE  IF   E           K DISK

      /copy socket_h
      /copy gskssl_h
      /copy spawn_h
      /copy signal_h
      /copy errno_h

      *
      * FIXME: These must match the location where you compile
      *        this object.
      *
     D PROGRAM         C                   'SSLSERVER'
     D LIBRARY         C                   'LIBSCK'

     D SSLSERVER       PR
     D   Instance                     1A   options(*nopass)
     D SSLSERVER       PI
     D   Instance                     1A   options(*nopass)

     D ListenForConn   PR
     D SpawnJob        PR
     D   client                      10I 0 value
     D HandleClient    PR
     D CreateEnv       PR                  like(gsk_handle)
     D ConnSock        PR            10I 0
     d   host                       256A   const
     D   port                        10I 0 value
     D UpgradeSock     PR                  like(gsk_handle)
     D    SslEnv                           like(gsk_handle) value
     D    sock                       10I 0 value
     D GetCustInfo     PR
     D    SslSock                          like(gsk_handle) value
     D CloseSsl        PR
     D    Handle                           like(gsk_handle) value
     D    SslEnv                           like(gsk_handle) value

     D errno           s             10I 0 based(p_errno)
     D ReportError     PR
     D EscapeMsg       PR
     D init_signals    PR
     D got_alarm       PR
     D   signo                       10I 0 value
     D errMsg          s             80A   varying

      /free

         init_signals();

         if ( %parms < 1 or Instance='N' );
             ListenForConn();
         else;
             HandleClient();
         endif;

         *inlr = *on;
      /end-free


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      *  ListenForConn(): Listen for new connections to be made
      *                   Each time a new TCP connection is made,
      *                   this spawns a new copy of itself to handle
      *                   that connection
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P ListenForConn   B
     D ListenForConn   PI

     D s               s             10I 0 inz(1)
     D client          s             10I 0 inz(1)
     D on              s             10I 0 inz(1)
     D bindto          ds                  likeds(sockaddr_in)

      /free

         // Create a socket

         s = socket(AF_INET: SOCK_STREAM: IPPROTO_IP);
         if (s < 0);
            ReportError();
         endif;

         // Allow the port to be re-used without waiting for 2
         // minutes first.

         setsockopt(s: SOL_SOCKET: SO_REUSEADDR: %addr(on): %size(on));


         // Bind the socket to port number 8765

         bindto = *ALLx'00';
         bindto.sin_family = AF_INET;
         bindto.sin_port   = 8765;

         if (bind(s: %addr(bindto): %size(bindto)) = -1);
            callp close(s);
            ReportError();
         endif;

         // Tell the socket that we're willing to receive
         //  incoming connections on this port

         if (listen(s: 32) = -1);
            callp close(s);
            ReportError();
         endif;

         // Receive connections and spawn a new job for each one
         //   use alarm() so that %shtdn() can be checked every
         //   15 seconds.  errno=EINTR means that the alarm went off.

         dou %shtdn();

            alarm(15);
            client = accept(s: *NULL: *OMIT);
            alarm(0);

            if (client = -1);
               p_errno = sys_errno();
               if (errno <> EINTR);
                 leave;
               endif;
            else;
               spawnJob(client);
            endif;
         enddo;

         callp close(s);
         return;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      *  Create a new job running another copy of this program,
      *  and have it take control of the connected socket
      *
      * Note: the spawn() API works very similarly to the SBMJOB
      *       command, except that:
      *        a) It uses IFS-style naming instead of traditional
      *              library/object naming.
      *        b) It lets you inherit socket descriptors so they
      *              can be used in the newly created job.
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P SpawnJob        B
     D SpawnJob        PI
     D   client                      10I 0 value

     D envvar          s            100A   dim(1)
     D path            s            100A
     D parm            s              2A
     D argv            s               *   dim(3) inz(*null)
     D envp            s               *   dim(2) inz(*null)
     D inh             ds                  likeds(Inheritance_T)
     D fdmap           s             10I 0 dim(1)
     D pid             s                   like(pid_t)
      /free

         envvar(1) = 'QIBM_USE_DESCRIPTOR_STDIO=Y' + x'00';
         envp(1)   = %addr(envvar(1));
         envp(2)   = *NULL;
         fdmap(1)  = client;
         path      = '/QSYS.LIB/'+LIBRARY+'.LIB/'+PROGRAM+'.PGM'+x'00';
         parm      = 'Y' + x'00';
         argv(1)   = %addr(path);
         argv(2)   = %addr(parm);
         argv(3)   = *NULL;
         inh       = *ALLx'00';

         pid = spawn( argv(1)
                    : 1
                    : fdmap
                    : inh
                    : argv
                    : envp
                    );

         callp close(client);
         return;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      *  Create a new job running another copy of this program,
      *  and have it take control of the connected socket
      *
      *  Because the spawnJob() routine maps the socket descriptor
      *  (no matter what it was) to be zero in this job, we'll
      *  always read & write from descriptor 0.
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P HandleClient    B
     D HandleClient    PI

     D env             s                   like(gsk_handle)
     D SslSock         s                   like(gsk_handle)

      /free

         // Create an SSL environment

         env = CreateEnv();
         if (env = *NULL);
            EscapeMsg();
         endif;

         // Upgrade the socket to SSL

         SSLSock = UpgradeSock(env: 0);
         if (SSLSock = *NULL);
            EscapeMsg();
         endif;

         // Handle customer info requests

         GetCustInfo(SslSock);

         // Shut down

         CloseSsl(SslSock:Env);

      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * CreateEnv(): Create an SSL environment for client sockets
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P CreateEnv       B
     D CreateEnv       PI                  like(gsk_handle)
     D rc              s             10I 0
     D SslEnv          s                   like(Gsk_handle)
      /free

        // Create an SSL environment with default values:

         rc = gsk_environment_open(SslEnv);
         if (rc <> GSK_OK);
            errMsg = %str(gsk_strerror(rc));
            return *NULL;
         endif;

        // Tell the environment to use the KLEMENT_SSLDEMO_SSLSERVER
        // application profile

         rc = gsk_attribute_set_buffer( SslEnv
                                      : GSK_OS400_APPLICATION_ID
                                      : 'KLEMENT_SSLDEMO_SSLSERVER'
                                      : 0 );
         if (rc <> GSK_OK);
            errMsg = %str(gsk_strerror(rc));
            gsk_environment_close( SslEnv );
            return *NULL;
         endif;

        // Tell the environment that this is a server connection

         rc = gsk_attribute_set_enum( SslEnv
                                    : GSK_SESSION_TYPE
                                    : GSK_SERVER_SESSION );
         if (rc <> GSK_OK);
            errMsg = %str(gsk_strerror(rc));
            gsk_environment_close( SslEnv );
            return *NULL;
         endif;

        // Activate the new environment.

         rc = gsk_environment_init( SslEnv );
         if (rc <> GSK_OK);
            errMsg = %str(gsk_strerror(rc));
            gsk_environment_close( SslEnv );
            return *NULL;
         endif;

         return SslEnv;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * UpgradeSock():  Upgrade a socket to use SSL
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P UpgradeSock     B
     D UpgradeSock     PI                  like(gsk_handle)
     D    SslEnv                           like(gsk_handle) value
     D    sock                       10I 0 value
     D Handle          s                   like(Gsk_handle)
     D rc              s             10I 0
      /free
          rc = gsk_secure_soc_open(SslEnv: Handle);
          if (rc <> GSK_OK);
             errMsg = %str(gsk_strerror(rc));
             return *NULL;
          endif;

          rc = gsk_attribute_set_numeric_value( Handle
                                              : GSK_HANDSHAKE_TIMEOUT
                                              : 30 );
          if (rc <> GSK_OK);
             errMsg = %str(gsk_strerror(rc));
             gsk_secure_soc_close(Handle);
             return *NULL;
          endif;

          rc = gsk_attribute_set_numeric_value( Handle
                                              : GSK_FD
                                              : sock );
          if (rc <> GSK_OK);
             errMsg = %str(gsk_strerror(rc));
             gsk_secure_soc_close(Handle);
             return *NULL;
          endif;

          rc = gsk_secure_soc_init( Handle );
          if (rc <> GSK_OK);
             errMsg = %str(gsk_strerror(rc));
             gsk_secure_soc_close(Handle);
             return *NULL;
          endif;

          return Handle;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      *  GetCustInfo():  retrieve the customer info for the client
      *                  and send it back...
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P GetCustInfo     B
     D GetCustInfo     PI
     D    SslSock                          like(gsk_handle) value

     D Req             ds                  qualified
     D   cmd                          4A
     D   CustNo                       4S 0

     D CustData        ds                  qualified
     D   Error                        1N   inz(*OFF)
     D   ErrorMsg                    60A
     D   Name                        25A
     D   Street                      25A
     D   City                        25A
     D   State                        2A
     D   Zip                         10A

     D totalRecv       s             10I 0
     D rc              s             10I 0
     D len             s             10I 0
     D bytesSent       s             10I 0
      /free

         dou %shtdn();

            reset CustData;

            //  Receive customer number

            totalRecv = 0;
            dou totalRecv = %size(Req);

                rc = gsk_secure_soc_read( SSLSock
                                        : %addr(Req) + totalRecv
                                        : %size(Req) - totalRecv
                                        : len );
                if (rc <> GSK_OK);
                   return;
                endif;

                totalRecv = totalRecv + len;
            enddo;

            // The command is for future expansion.  If I wanted to have
            // other data that this program could return, I could have
            // other commands that have different responses.

            if (Req.Cmd <> 'CUST');
               return;
            endif;

            chain Req.CustNo CUSTFILE;
            if not %found;
               CustData.error    = *ON;
               CustData.ErrorMsg = 'Customer number not found';
            else;
               CustData.error  = *OFF;
               CustData.Name   = Name;
               CustData.Street = Street;
               CustData.City   = City;
               CustData.State  = State;
               CustData.Zip    = Zip;
            endif;

            rc = gsk_secure_soc_write( SSLSock
                                     : %addr(CustData)
                                     : %size(CustData)
                                     : bytesSent );
            if (rc <> GSK_OK);
                return;
            endif;

         enddo;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * CloseSsl():  Close an SSL socket
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P CloseSsl        B
     D CloseSsl        PI
     D    Handle                           like(gsk_handle) value
     D    SslEnv                           like(gsk_handle) value
      /free
           gsk_secure_Soc_close( handle);
           gsk_environment_close( SslEnv );
           callp close(0);
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * EscapeMsg(): Send an escape message w/reason for SSL failure
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P EscapeMsg       B
     D EscapeMsg       PI

     D SndPgmMsg       PR                  ExtPgm('QMHSNDPM')
     D   MessageID                    7A   Const
     D   QualMsgF                    20A   Const
     D   MsgData                    256A   Const
     D   MsgDtaLen                   10I 0 Const
     D   MsgType                     10A   Const
     D   CallStkEnt                  10A   Const
     D   CallStkCnt                  10I 0 Const
     D   MessageKey                   4A
     D   ErrorCode                    1A

     D ErrorCode       DS
     D  BytesProv                    10I 0 inz(0)
     D  BytesAvail                   10I 0 inz(0)

     D wwTheKey        S              4A
      /free

           SndPgmMsg( 'CPF9897'
                    : 'QCPFMSG   *LIBL'
                    : errMsg
                    : %len(%trimr(errMsg))
                    : '*ESCAPE'
                    : '*CTLBDY'
                    : 1
                    : wwTheKey
                    : ErrorCode );

      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * ReportError():  Send an escape message explaining any errors
      *                 that occurred.
      *
      *  This function requires binding directory QC2LE in order
      *  to access the __errno() function.
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P ReportError     B
     D ReportError     PI

     D QMHSNDPM        PR                  ExtPgm('QMHSNDPM')
     D   MessageID                    7A   Const
     D   QualMsgF                    20A   Const
     D   MsgData                      1A   Const
     D   MsgDtaLen                   10I 0 Const
     D   MsgType                     10A   Const
     D   CallStkEnt                  10A   Const
     D   CallStkCnt                  10I 0 Const
     D   MessageKey                   4A
     D   ErrorCode                 8192A   options(*varsize)

     D ErrorCode       DS                  qualified
     D  BytesProv              1      4I 0 inz(0)
     D  BytesAvail             5      8I 0 inz(0)

     D MsgKey          S              4A
     D MsgID           s              7A

      /free

         p_errno = sys_errno();
         MsgID = 'CPE' + %char(errno);

         QMHSNDPM( MsgID
                 : 'QCPFMSG   *LIBL'
                 : ' '
                 : 0
                 : '*ESCAPE'
                 : '*PGMBDY'
                 : 1
                 : MsgKey
                 : ErrorCode         );

      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * Set up a signal handler to receive the SIGALRM signal
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P init_signals    B
     D init_signals    PI
     D act             ds                  likeds(sigaction_t)
      /free
          sigfillset(act.sa_mask);
          act.sa_handler   = %paddr(got_alarm);
          act.sa_flags     = 0;
          act.sa_sigaction = *NULL;

          sigaction(SIGALRM: act: *omit);
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * Whenever this program receives a SIGALRM signal, this
      * subprocedure will be called by the system
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P got_alarm       B
     D got_alarm       PI
     D   signo                       10I 0 value
      /free
         // Do nothing. The connect() API will return
         //  EINTR ("interrupted by signal") when the
         //  signal is received.
      /end-free
     P                 E
